<?php
require_once '../config/config.php';

if (!isAdmin()) {
    redirect('../dashboard.php');
}

$db = Database::getInstance()->getConnection();

// Handle API config save
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'save_config') {
    $api_url = sanitize($_POST['api_url'] ?? '');
    $api_key = sanitize($_POST['api_key'] ?? '');
    $api_secret = sanitize($_POST['api_secret'] ?? '');
    $sender_number = sanitize($_POST['sender_number'] ?? '');
    
    // Check if config exists
    $stmt = $db->query("SELECT id FROM wa_config LIMIT 1");
    $existing = $stmt->fetch();
    
    if ($existing) {
        $stmt = $db->prepare("UPDATE wa_config SET api_url = ?, api_key = ?, api_secret = ?, sender_number = ? WHERE id = ?");
        $stmt->execute([$api_url, $api_key, $api_secret, $sender_number, $existing['id']]);
    } else {
        $stmt = $db->prepare("INSERT INTO wa_config (api_url, api_key, api_secret, sender_number) VALUES (?, ?, ?, ?)");
        $stmt->execute([$api_url, $api_key, $api_secret, $sender_number]);
    }
    
    $_SESSION['success'] = 'Konfigurasi WhatsApp API berhasil disimpan';
    redirect('wa_broadcast.php');
}

// Handle broadcast
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'broadcast') {
    $judul = sanitize($_POST['judul'] ?? '');
    $pesan = sanitize($_POST['pesan'] ?? '');
    $filter_desa_id = intval($_POST['filter_desa_id'] ?? 0);
    $filter_dusun_id = intval($_POST['filter_dusun_id'] ?? 0);
    $filter_status = sanitize($_POST['filter_status'] ?? '');
    
    // Handle image upload
    $gambar = '';
    if (isset($_FILES['gambar']) && $_FILES['gambar']['error'] === UPLOAD_ERR_OK) {
        $upload_dir = '../uploads/broadcast/';
        if (!is_dir($upload_dir)) {
            mkdir($upload_dir, 0755, true);
        }
        
        $file_ext = strtolower(pathinfo($_FILES['gambar']['name'], PATHINFO_EXTENSION));
        $allowed_ext = ['jpg', 'jpeg', 'png', 'gif'];
        
        if (in_array($file_ext, $allowed_ext)) {
            $filename = 'broadcast_' . time() . '_' . uniqid() . '.' . $file_ext;
            $target_path = $upload_dir . $filename;
            
            if (move_uploaded_file($_FILES['gambar']['tmp_name'], $target_path)) {
                $gambar = 'uploads/broadcast/' . $filename;
            }
        }
    }
    
    // Build filter query
    $where = "u.role = 'warga' AND u.status = 'active'";
    if ($filter_desa_id > 0) {
        $where .= " AND des.id = $filter_desa_id";
    }
    if ($filter_dusun_id > 0) {
        $where .= " AND d.id = $filter_dusun_id";
    }
    if (!empty($filter_status)) {
        $where .= " AND u.status = '$filter_status'";
    }
    
    // Get target users
    $stmt = $db->query("SELECT u.*, d.nama_dusun, des.nama_desa 
                       FROM users u 
                       LEFT JOIN dusun d ON u.dusun_id = d.id 
                       LEFT JOIN desa des ON d.desa_id = des.id 
                       WHERE $where AND u.no_hp IS NOT NULL AND u.no_hp != ''");
    $target_users = $stmt->fetchAll();
    $total_target = count($target_users);
    
    if ($total_target === 0) {
        $_SESSION['error'] = 'Tidak ada warga yang memenuhi filter';
        redirect('wa_broadcast.php');
    }
    
    // Create broadcast record
    $stmt = $db->prepare("INSERT INTO wa_broadcast (judul, pesan, gambar, filter_desa_id, filter_dusun_id, filter_status, total_target, created_by, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'pending')");
    $stmt->execute([$judul, $pesan, $gambar, $filter_desa_id > 0 ? $filter_desa_id : null, $filter_dusun_id > 0 ? $filter_dusun_id : null, $filter_status, $total_target, $_SESSION['user_id']]);
    $broadcast_id = $db->lastInsertId();
    
    // Get WA config
    $stmt = $db->query("SELECT * FROM wa_config WHERE is_active = 1 LIMIT 1");
    $wa_config = $stmt->fetch();
    
    if (!$wa_config || empty($wa_config['api_url'])) {
        $_SESSION['error'] = 'Konfigurasi WhatsApp API belum diatur. Silakan atur terlebih dahulu.';
        redirect('wa_broadcast.php');
    }
    
    // Update status to sending
    $db->prepare("UPDATE wa_broadcast SET status = 'sending' WHERE id = ?")->execute([$broadcast_id]);
    
    // Send messages
    $success_count = 0;
    $failed_count = 0;
    
    foreach ($target_users as $user) {
        $phone = preg_replace('/[^0-9]/', '', $user['no_hp']); // Remove non-numeric
        if (substr($phone, 0, 1) === '0') {
            $phone = '62' . substr($phone, 1); // Convert to international format
        } elseif (substr($phone, 0, 2) !== '62') {
            $phone = '62' . $phone;
        }
        
        $result = sendWhatsAppMessage($wa_config, $phone, $judul, $pesan, $gambar);
        
        // Log result
        $log_stmt = $db->prepare("INSERT INTO wa_broadcast_log (broadcast_id, user_id, phone_number, status, error_message) VALUES (?, ?, ?, ?, ?)");
        if ($result['success']) {
            $log_stmt->execute([$broadcast_id, $user['id'], $phone, 'success', null]);
            $success_count++;
        } else {
            $log_stmt->execute([$broadcast_id, $user['id'], $phone, 'failed', $result['error']]);
            $failed_count++;
        }
    }
    
    // Update broadcast status
    $status = ($failed_count === 0) ? 'completed' : (($success_count > 0) ? 'completed' : 'failed');
    $db->prepare("UPDATE wa_broadcast SET status = ?, total_terkirim = ?, total_gagal = ? WHERE id = ?")
       ->execute([$status, $success_count, $failed_count, $broadcast_id]);
    
    $_SESSION['success'] = "Broadcast berhasil! Terkirim: $success_count, Gagal: $failed_count dari $total_target warga";
    redirect('wa_broadcast.php');
}

// Function to send WhatsApp message
function sendWhatsAppMessage($config, $phone, $judul, $pesan, $gambar = '') {
    $api_url = rtrim($config['api_url'], '/');
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $api_url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    
    // If image exists, use multipart/form-data
    if (!empty($gambar) && file_exists('../' . $gambar)) {
        $data = [
            'phone' => $phone,
            'message' => "*$judul*\n\n$pesan",
            'image' => new CURLFile('../' . $gambar),
            'api_key' => $config['api_key'] ?? '',
            'api_secret' => $config['api_secret'] ?? ''
        ];
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    } else {
        $data = [
            'phone' => $phone,
            'message' => "*$judul*\n\n$pesan",
            'api_key' => $config['api_key'] ?? '',
            'api_secret' => $config['api_secret'] ?? ''
        ];
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . ($config['api_key'] ?? '')
        ]);
    }
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        return ['success' => false, 'error' => $error];
    }
    
    if ($http_code >= 200 && $http_code < 300) {
        return ['success' => true];
    } else {
        return ['success' => false, 'error' => "HTTP $http_code: " . substr($response, 0, 100)];
    }
}

// Get WA config
$stmt = $db->query("SELECT * FROM wa_config WHERE is_active = 1 LIMIT 1");
$wa_config = $stmt->fetch();

// Get desa and dusun for filter
$desa_list = $db->query("SELECT * FROM desa ORDER BY nama_desa")->fetchAll();
$dusun_list = $db->query("SELECT d.*, des.nama_desa FROM dusun d JOIN desa des ON d.desa_id = des.id ORDER BY des.nama_desa, d.nama_dusun")->fetchAll();

// Get broadcast history
$stmt = $db->query("SELECT b.*, u.nama_lengkap as created_by_name 
                   FROM wa_broadcast b 
                   JOIN users u ON b.created_by = u.id 
                   ORDER BY b.created_at DESC 
                   LIMIT 20");
$broadcast_history = $stmt->fetchAll();

?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Broadcast WhatsApp - <?php echo APP_NAME; ?></title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <?php include '../includes/sidebar.php'; ?>
    
    <main class="main-content">
        <header class="content-header">
            <div class="header-left">
                <h1><i class="fab fa-whatsapp"></i> Broadcast WhatsApp</h1>
                <p>Kirim peringatan darurat ke warga melalui WhatsApp</p>
            </div>
        </header>

        <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success">
            <i class="fas fa-check-circle"></i> <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
        </div>
        <?php endif; ?>

        <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger">
            <i class="fas fa-exclamation-circle"></i> <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
        </div>
        <?php endif; ?>

        <!-- API Config -->
        <div class="content-section">
            <h2><i class="fas fa-cog"></i> Konfigurasi WhatsApp API</h2>
            <form method="POST" class="form-container">
                <input type="hidden" name="action" value="save_config">
                <div class="alert alert-info">
                    <i class="fas fa-info-circle"></i> 
                    <strong>Catatan:</strong> Masukkan URL API WhatsApp Anda. Format umum: 
                    <code>https://api.whatsapp.com/send</code> atau 
                    <code>https://your-api-domain.com/api/send</code>
                </div>
                <div class="form-group">
                    <label>API URL</label>
                    <input type="url" name="api_url" value="<?php echo htmlspecialchars($wa_config['api_url'] ?? ''); ?>" placeholder="https://api.example.com/send" required>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>API Key</label>
                        <input type="text" name="api_key" value="<?php echo htmlspecialchars($wa_config['api_key'] ?? ''); ?>" placeholder="Masukkan API Key">
                    </div>
                    <div class="form-group">
                        <label>API Secret</label>
                        <input type="password" name="api_secret" value="<?php echo htmlspecialchars($wa_config['api_secret'] ?? ''); ?>" placeholder="Masukkan API Secret">
                    </div>
                </div>
                <div class="form-group">
                    <label>Nomor Pengirim</label>
                    <input type="text" name="sender_number" value="<?php echo htmlspecialchars($wa_config['sender_number'] ?? ''); ?>" placeholder="Contoh: 6281234567890">
                    <small style="color: #666;">Format: 62xxxxxxxxxxx (tanpa + dan 0 di depan)</small>
                </div>
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-save"></i> Simpan Konfigurasi
                </button>
            </form>
        </div>

        <!-- Broadcast Form -->
        <div class="content-section">
            <h2><i class="fas fa-paper-plane"></i> Buat Broadcast Peringatan</h2>
            <form method="POST" enctype="multipart/form-data" class="form-container" id="broadcastForm">
                <input type="hidden" name="action" value="broadcast">
                
                <div class="form-group">
                    <label>Judul Peringatan <span style="color: red;">*</span></label>
                    <input type="text" name="judul" required placeholder="Contoh: Peringatan Kekeringan Darurat">
                </div>
                
                <div class="form-group">
                    <label>Pesan <span style="color: red;">*</span></label>
                    <textarea name="pesan" rows="6" required placeholder="Tuliskan pesan peringatan yang akan dikirim ke warga..."></textarea>
                    <small style="color: #666;">Pesan akan dikirim dengan format WhatsApp</small>
                </div>
                
                <div class="form-group">
                    <label>Gambar (Opsional)</label>
                    <input type="file" name="gambar" accept="image/*" id="gambar_input">
                    <div id="gambar_preview" style="margin-top: 10px; display: none;">
                        <img id="preview_img" src="" alt="Preview" style="max-width: 100%; max-height: 300px; border-radius: 8px;">
                        <button type="button" class="btn btn-sm btn-danger" onclick="removeGambar()" style="margin-top: 10px;">
                            <i class="fas fa-times"></i> Hapus Gambar
                        </button>
                    </div>
                </div>
                
                <div class="form-group">
                    <label><i class="fas fa-filter"></i> Filter Warga</label>
                    <div class="alert alert-info" style="margin-bottom: 15px;">
                        <i class="fas fa-info-circle"></i> Pilih filter untuk menentukan warga yang akan menerima pesan. Kosongkan untuk mengirim ke semua warga.
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label>Filter Berdasarkan Desa</label>
                        <select name="filter_desa_id" id="filter_desa" onchange="updateDusunFilter()">
                            <option value="">Semua Desa</option>
                            <?php foreach ($desa_list as $desa): ?>
                            <option value="<?php echo $desa['id']; ?>"><?php echo htmlspecialchars($desa['nama_desa']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <label>Filter Berdasarkan Dusun</label>
                        <select name="filter_dusun_id" id="filter_dusun">
                            <option value="">Semua Dusun</option>
                            <?php foreach ($dusun_list as $dusun): ?>
                            <option value="<?php echo $dusun['id']; ?>" data-desa-id="<?php echo $dusun['desa_id']; ?>" style="display: none;">
                                <?php echo htmlspecialchars($dusun['nama_desa']); ?> - <?php echo htmlspecialchars($dusun['nama_dusun']); ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                
                <div class="form-group">
                    <label>Filter Berdasarkan Status</label>
                    <select name="filter_status">
                        <option value="">Semua Status</option>
                        <option value="active">Aktif</option>
                        <option value="inactive">Nonaktif</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <div id="target_count" class="alert alert-warning" style="display: none;">
                        <i class="fas fa-users"></i> <strong>Estimasi Target:</strong> <span id="target_number">0</span> warga
                    </div>
                </div>
                
                <button type="submit" class="btn btn-success btn-block" onclick="return confirm('Yakin ingin mengirim broadcast ke warga yang terfilter?')">
                    <i class="fab fa-whatsapp"></i> Kirim Broadcast
                </button>
            </form>
        </div>

        <!-- Broadcast History -->
        <div class="content-section">
            <h2><i class="fas fa-history"></i> Riwayat Broadcast</h2>
            <div class="table-responsive">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Tanggal</th>
                            <th>Judul</th>
                            <th>Target</th>
                            <th>Terkirim</th>
                            <th>Gagal</th>
                            <th>Status</th>
                            <th>Dibuat Oleh</th>
                            <th>Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($broadcast_history)): ?>
                        <tr>
                            <td colspan="8" class="text-center">Belum ada broadcast</td>
                        </tr>
                        <?php else: ?>
                        <?php foreach ($broadcast_history as $bc): ?>
                        <tr>
                            <td><?php echo formatDateTime($bc['created_at']); ?></td>
                            <td><?php echo htmlspecialchars($bc['judul']); ?></td>
                            <td><?php echo $bc['total_target']; ?></td>
                            <td><span class="badge badge-success"><?php echo $bc['total_terkirim']; ?></span></td>
                            <td><span class="badge badge-danger"><?php echo $bc['total_gagal']; ?></span></td>
                            <td>
                                <span class="badge badge-<?php 
                                    echo $bc['status'] === 'completed' ? 'success' : 
                                        ($bc['status'] === 'sending' ? 'warning' : 
                                        ($bc['status'] === 'failed' ? 'danger' : 'info')); 
                                ?>">
                                    <?php echo ucfirst($bc['status']); ?>
                                </span>
                            </td>
                            <td><?php echo htmlspecialchars($bc['created_by_name']); ?></td>
                            <td>
                                <a href="wa_broadcast_detail.php?id=<?php echo $bc['id']; ?>" class="btn btn-sm btn-primary">
                                    <i class="fas fa-eye"></i> Detail
                                </a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </main>

    <script src="../assets/js/main.js"></script>
    <script>
    // Preview gambar
    document.getElementById('gambar_input').addEventListener('change', function(e) {
        const file = e.target.files[0];
        if (file) {
            const reader = new FileReader();
            reader.onload = function(e) {
                document.getElementById('preview_img').src = e.target.result;
                document.getElementById('gambar_preview').style.display = 'block';
            };
            reader.readAsDataURL(file);
        }
    });
    
    function removeGambar() {
        document.getElementById('gambar_input').value = '';
        document.getElementById('gambar_preview').style.display = 'none';
    }
    
    function updateDusunFilter() {
        const desaId = document.getElementById('filter_desa').value;
        const dusunSelect = document.getElementById('filter_dusun');
        const options = dusunSelect.querySelectorAll('option[data-desa-id]');
        
        // Reset dusun selection
        dusunSelect.value = '';
        
        options.forEach(option => {
            if (!desaId || option.getAttribute('data-desa-id') === desaId) {
                option.style.display = '';
            } else {
                option.style.display = 'none';
            }
        });
        
        updateTargetCount();
    }
    
    function countTarget() {
        const desaId = document.getElementById('filter_desa').value;
        const dusunId = document.getElementById('filter_dusun').value;
        const status = document.querySelector('select[name="filter_status"]').value;
        
        // Show loading
        document.getElementById('target_count').style.display = 'block';
        document.getElementById('target_number').textContent = 'Menghitung...';
        
        // Make AJAX call
        const formData = new FormData();
        formData.append('action', 'count_target');
        if (desaId) formData.append('desa_id', desaId);
        if (dusunId) formData.append('dusun_id', dusunId);
        if (status) formData.append('status', status);
        
        fetch('wa_broadcast_count.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            document.getElementById('target_number').textContent = data.count || 0;
        })
        .catch(error => {
            document.getElementById('target_number').textContent = 'Error';
            console.error('Error:', error);
        });
    }
    
    function updateTargetCount() {
        // Auto hide if all filters cleared
        const desaId = document.getElementById('filter_desa').value;
        const dusunId = document.getElementById('filter_dusun').value;
        const status = document.querySelector('select[name="filter_status"]').value;
        
        if (!desaId && !dusunId && !status) {
            document.getElementById('target_count').style.display = 'none';
        }
    }
    
    document.getElementById('filter_dusun').addEventListener('change', updateTargetCount);
    document.querySelector('select[name="filter_status"]').addEventListener('change', updateTargetCount);
    </script>
</body>
</html>

